#ifndef H_CDW_TASK
#define H_CDW_TASK

#include <stdbool.h>
#include <signal.h> /* sig_atomic_t */

#include "main.h"
#include "cdw_disc.h"
#include "cdw_config.h"
#include "cdw_iso9660.h"
#include "cdw_udf.h"
#include "cdw_ext_tools.h"

/* forward declaration */
typedef struct cdw_task cdw_task_t;

cdw_task_t *cdw_task_new(cdw_id_t id, cdw_disc_t *disc);
void        cdw_task_save_options(cdw_task_t *task);
void        cdw_task_delete(cdw_task_t **task);
cdw_rv_t    cdw_task_copy(cdw_task_t *target, cdw_task_t *source);


cdw_rv_t cdw_task_check_tool_status(cdw_task_t *task);
void     cdw_task_reset_tool_status(cdw_task_t *task);
cdw_rv_t cdw_task_select_tools_for_task(cdw_task_t *task, cdw_disc_t *disc);

const char *cdw_task_get_session_mode_label(cdw_id_t mode);
const char *cdw_task_get_disc_mode_label(cdw_id_t mode);

void cdw_task_debug_print_burn_options(cdw_task_t *task);
void cdw_task_debug_print_options(cdw_task_t *task, const char *point);





#define CDW_TOOL_STATUS_OK                                       0x00000000

/* mkisofs */
#define CDW_TOOL_STATUS_MKISOFS_THE_SAME_JOLIET_NAME             0x00000001
#define CDW_TOOL_STATUS_MKISOFS_JOLIET_TREE_SORT                 0x00000002
#define CDW_TOOL_STATUS_MKISOFS_CANT_FIND_BOOT_IMAGE             0x00000004
#define CDW_TOOL_STATUS_MKISOFS_NO_SUCH_FILE_OR_DIR              0x00000008
#define CDW_TOOL_STATUS_MKISOFS_BAD_OPTION                       0x00000010
#define CDW_TOOL_STATUS_MKISOFS_INODE_NUMBER_OVERFLOW            0x00000020
#define CDW_TOOL_STATUS_MKISOFS_NO_SPACE_LEFT                    0x00000040
#define CDW_TOOL_STATUS_MKISOFS_ISO_LEVEL_MISMATCH               0x00000080
#define CDW_TOOL_STATUS_MKISOFS_THE_SAME_RR_NAME                 0x00000100
#define CDW_TOOL_STATUS_MKISOFS_FILE_DID_GROW                    0x00000200
#define CDW_TOOL_STATUS_MKISOFS_FATAL_ERROR                      0x00000300
#define CDW_TOOL_STATUS_MKISOFS_ALLOW_LIMITED_SIZE               0x00000800
#define CDW_TOOL_STATUS_MKISOFS_LOW_ISO_LEVEL                    0x00001000
#define CDW_TOOL_STATUS_MKISOFS_SHORT_READ                       0x00002000
#define CDW_TOOL_STATUS_MKISOFS_IMPLEMENTATION_BOTCH             0x00004000

/* cdrecord */
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_DEVICE              0x00000001
#define CDW_TOOL_STATUS_CDRECORD_NO_SUCH_FILE_OR_DIR             0x00000002
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_SEND_CUE_SHEET           0x00000004
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_BLANK_DVD_RWP_NON_RICOH  0x00000008
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_NEXT                0x00000010
#define CDW_TOOL_STATUS_CDRECORD_ERROR_AFTER                     0x00000020
#define CDW_TOOL_STATUS_CDRECORD_DATA_MAY_NOT_FIT                0x00000040
#define CDW_TOOL_STATUS_CDRECORD_BAD_OPTION                      0x00000080
#define CDW_TOOL_STATUS_CDRECORD_NO_DISK_WRONG_DISK              0x00000100

/* growisofs */
#define CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_FILES             0x00000001
#define CDW_TOOL_STATUS_GROWISOFS_MEDIA_NOT_APPENDABLE           0x00000002
#define CDW_TOOL_STATUS_GROWISOFS_USE_Z_OPTION                   0x00000004
#define CDW_TOOL_STATUS_GROWISOFS_IO_ERROR                       0x00000008
#define CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_IMAGE             0x00000010
#define CDW_TOOL_STATUS_GROWISOFS_4GB_BOUNDARY                   0x00000020


/* in the process of verification there may be used two different tools:
   standard code reading files, or library code (from libcdio
   or similar library) reading optical disc; in the first case
   there should be displayed some generic system message (e.g.
   can't find some file in user's system), in the second case
   there should be displayed information about problems with
   accessing optical disc or about problems with optical drive */
#define CDW_TOOL_STATUS_CALCULATE_DIGEST_SETUP_FAIL              0x00000001
#define CDW_TOOL_STATUS_CALCULATE_DIGEST_READ_FAIL               0x00000002
#define CDW_TOOL_STATUS_CALCULATE_DIGEST_WRONG_TRACK             0x00000004

/* libcdio */
#define CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL                       0x00080001
#define CDW_TOOL_STATUS_LIBCDIO_READ_FAIL                        0x00080002

/* xorriso */
#define CDW_TOOL_STATUS_XORRISO_SIMULATION_NOT_SUPPORTED         0x00000001
#define CDW_TOOL_STATUS_XORRISO_NOT_ENOUGH_SPACE                 0x00000002
#define CDW_TOOL_STATUS_XORRISO_CANNOT_READ_ISO_IMAGE_TREE       0x00000004
#define CDW_TOOL_STATUS_XORRISO_NOT_A_KNOWN_OPTION               0x00000008

/* mkudffs and its helpers. */

/* Segmentation fault may happen to any tool, but it's most likely to happen to mkudffs. */
#define CDW_TOOL_STATUS_MKUDFFS_GENERAL_SEGMENTATION_FAULT       0x00000001
#define CDW_TOOL_STATUS_SUDO_PASSWORD_PROMPT                     0x00000001
#define CDW_TOOL_STATUS_MOUNT_FAILURE                            0x00000001
#define CDW_TOOL_STATUS_UMOUNT_NOT_MOUNTED                       0x00000001
#define CDW_TOOL_STATUS_RSYNC_NO_SPACE_LEFT                      0x00000001



enum task_id {
	CDW_TASK_NONE,               /**< no task currently assigned */

	CDW_TASK_BURN_FROM_IMAGE,    /**< burn image to optical disc */
	CDW_TASK_BURN_FROM_FILES,    /**< burn files to optical disc */
	CDW_TASK_READ_DISC,          /**< read content of optical disc */
	CDW_TASK_ERASE_DISC,         /**< erase optical disc - this includes formatting */
	//CDW_TASK_CLONE_DISC,         /**< clone one optical disc to another optical disc */

	CDW_TASK_CALCULATE_DIGEST,   /**< calculate digest of specified object */

	CDW_TASK_MEDIA_INFO,         /**< get check media information */

	CDW_TASK_CREATE_IMAGE_ISO9660,
	CDW_TASK_CREATE_IMAGE_UDF
};





struct cdw_task {

	cdw_id_t id;

	/* Full path to ISO9660 or UDF image file. Used when creating
	   new image from files, or when writing image to optical
	   disc. */
	char *image_file_fullpath;

	struct {
		bool ok;

		/* Most of actions are performed by external tools,
		   which are executed as child processes.
		   child_exit_status is set with wait(). This is a raw
		   value. Different tools may return different values
		   on success/failure. */
		sig_atomic_t child_exit_status_raw;
		int child_exit_status;

		size_t mkisofs;
		size_t cdrecord;
		size_t calculate_digest;
		size_t growisofs;
		size_t dvd_rw_mediainfo;
		size_t libcdio;
		size_t xorriso;

		/* UDF tools: mkudffs and its helpers. */
		size_t mkudffs_general;
		size_t mkudffs;
		size_t truncate;
		size_t sudo;
		size_t mount;
		size_t umount;
		size_t sync;
		size_t rsync;
	} tool_status;


	/* On some day I will rename this variable to "write". */
	cdw_write_t burn;


	cdw_erase_t erase;


	struct {
		cdw_id_label_t tool;

		/* Options related to creating ISO9660 file system. */
		cdw_iso9660_t iso9660;

		/* Options related to creating UDF file system. */
		cdw_udf_t udf;

	} create_image;



	struct {
		cdw_id_label_t tool;
	} media_info;

	struct {
		cdw_id_label_t tool;

		cdw_id_t mode;

		long long int data_size_sectors;

		/* size of file to be verified */
		long long int file_size;

		int source_fd;
		int target_fd;

		void * (*read_and_write)(void *task);

	} calculate_digest;

	cdw_disc_t *disc;

}; /** \brief Data structure describing task that is currently
       performed or is about to be performed */




#endif /* H_CDW_TASK */
